import numpy as np
from copy import deepcopy
from shapely.geometry import Polygon
from shapely.geometry import Point
from math import sqrt, acos, cos
from matplotlib import pyplot as plt

def get_occluded_area(ego_center, obstacle_center, obstacle_polygon, sensor_range):
    """
    Calculate the occluded area represented by the polygons
    Params:
        ego_center: ego vehicle center position
        obstacle_center: list of tuples, the center of the obstacle
        obstacle_polygon: list of Polygons, the polygon of the obstacle
        sensor_range: float, the range of the sensor
    Return:
        occluded_polygon: the occluded area represented by the polygon
    """
    vector = obstacle_center - ego_center
    distance = sqrt(vector[0]**2 + vector[1]**2)
    polygon_coords = list(obstacle_polygon.exterior.coords)[0:-1]
    angle_list = []
    ego_coord_vec_list = []
    ego_coord_dis_list = []

    for coord in polygon_coords:
        d = (obstacle_center[0] - ego_center[0]) * (coord[1] - ego_center[1]) - (obstacle_center[1] - ego_center[1]) * (coord[0] - ego_center[0])
        ego_coord_vec = coord - ego_center
        ego_coord_dis = sqrt(ego_coord_vec[0]**2 + ego_coord_vec[1]**2)
        angle = np.sign(d) * acos(np.sum(ego_coord_vec * vector) / (ego_coord_dis * distance))
        angle_list.append(angle)
        ego_coord_vec_list.append(ego_coord_vec)
        ego_coord_dis_list.append(ego_coord_dis)

    angle_list = np.array(angle_list)
    neg_index = np.argmin(angle_list)
    pos_index = np.argmax(angle_list)
    neg_ego_coord_vec = ego_coord_vec_list[neg_index]
    pos_ego_coord_vec = ego_coord_vec_list[pos_index]
    neg_ego_coord_dis = ego_coord_dis_list[neg_index]
    pos_ego_coord_dis = ego_coord_dis_list[pos_index]
    angle = angle_list[pos_index] - angle_list[neg_index]
    max_distance = max(ego_coord_dis_list)
    length = max(sensor_range, max_distance) / (cos(angle/2)) + 0.5
    extend_neg_coord = ego_center + neg_ego_coord_vec * length / neg_ego_coord_dis
    extend_pos_coord = ego_center + pos_ego_coord_vec * length / pos_ego_coord_dis
    polygon = Polygon([polygon_coords[neg_index], extend_neg_coord, extend_pos_coord, polygon_coords[pos_index]])
    occluded_polygon = polygon.difference(obstacle_polygon)
    return occluded_polygon


def get_visible_area(ego, object_list, sensor_range):
    """
    Params:
        ego: ego object
        obstacles: list of tuples, the center of the obstacles
        sensor_range: float, the range of the sensor
    Return:
        visible_area: the visible area
    """
    ego_polygon = [bbx_to_polygon(ego)]
    polygon_list = []
    for obj in object_list:
        polygon_list.append(bbx_to_polygon(obj))
    polygon_list = ego_polygon + polygon_list

    ego_center = np.array([ego.location.x, ego.location.y])
    initialize = False
    for obj in object_list:
        obstacle_center = np.array([obj.location.x, obj.location.y])
        obstacle_polygon = bbx_to_polygon(obj)
        occluded_polygon = get_occluded_area(ego_center, obstacle_center, obstacle_polygon, sensor_range)
        if not initialize:
            occluded_area = deepcopy(occluded_polygon)
            initialize = True
        else:
            occluded_area = deepcopy(occluded_area.union(occluded_polygon))
    polygon_list = polygon_list + [occluded_area]
    visible_area = Point(ego_center).buffer(sensor_range).difference(occluded_area)
    polygon_list = polygon_list + [visible_area]
    #plot_polygon(polygon_list)
    return visible_area

def is_visible(visible_area, target, threshold=0.4):
    """
    Check if the target object is visible.
    Params:
        visible_area: the visible area represented by the polygon
        target_object: the target object represented by the polygon
        threshold: the threshold of the visible ratio
    """
    target = bbx_to_polygon(target)
    area = visible_area.intersection(target).area
    visible_ratio = area / max(target.area, 0.1)
    return visible_ratio > threshold

def check_2d_visibility_shapely(ego, object_list, sensor_range, threshold=0.4):
    """
    Check the 2D visibility of the objects in the object_list
    Params:
        ego: the ego carla.BoundingBox object
        object_list: the list of the objects carla.BoundingBox object
        sensor_range: the range of the sensor
    Return:
        visibility of each object in the object_list: list of boolean
    """
    visibility = []
    visible_area = get_visible_area(ego, object_list, sensor_range)
    for obj in object_list:
        if is_visible(visible_area, obj, threshold):
            visibility.append(True)
        else:
            visibility.append(False)
    #print(visibility)
    assert len(visibility) == len(object_list)
    return visibility

def bbx_to_polygon(bounding_box):
    """
    Get the polygon from the bounding box
    Params:
        bounding_box:  carla.BoundingBox, the bounding box of the object
    Return:
        polygon: the polygon represented by the list of tuples
    """
    vertices = bounding_box.get_local_vertices()
    vertices = [vertices[0], vertices[2], vertices[6], vertices[4]]
    polygon = Polygon([(vertice.x, vertice.y) for vertice in vertices])
    return polygon

def plot_polygon(polygons):
    plt.figure()
    #plt.xlim(202+35, 192-35)
    #plt.ylim(300-10, 200+10)
    for polygon in polygons:
        x, y = polygon.exterior.xy
        plt.plot(y, x)
    plt.show()
